LIBRARY "app/scripts/ota-update.brs"
LIBRARY "app/scripts/util.brs"

REM ------------------------------
REM  ScreenCloud BrightSign Player
REM  version: 2.3.3
REM ------------------------------

REM  Diagnostic Web Server
REM  url: http://brightsign-<device-serial-number>.local
REM  user: admin
REM  password: password

REM  ssh
REM  > ssh brightsign@brightsign-<device-serial-number>.local
REM  > password
REM
REM  telnet
REM  > telnet brightsign-<device-serial-number>.local

REM  Devtools
REM  url: http://<ip-address-from-diagnostic-web-server>:3000/
REM

Sub Main()
    If NOT IsProduction() Then
        EnableSSHAndSFTP()
    End If

    ' enable web inspector
    inspector_value = "0"
    If NOT IsProduction() Then inspector_value = "1"

    htmlRegistry = CreateObject("roRegistrySection", "html")
    htmlRegistry.Write("disable-web-security", "1")
    ' Does not seem to be able to turn debugging off in production
    htmlRegistry.Write("enable_web_inspector", inspector_value)
    htmlRegistry.Write("mse-support", "1")
    htmlRegistry.Write("disable-hw-accelerated-video-decode", "0")
    ' https://linear.app/screencloud/issue/PLAYERS-1130/video-brightsign-series-5-video-playback-issue
    htmlRegistry.Write("use-brightsign-media-player", "0")
    ' Video playback issue workaround for Series 5 ^^^
    htmlRegistry.Write("disable-http-cache", "0")
    htmlRegistry.Flush()


    bestMode = CreateObject("roVideoMode").getBestMode("hdmi")
    LogX("Autorun", "Best HDMI mode: ", bestMode)
    if bestMode = "" then
        LogX("Autorun", "No best mode found (Display not connected?), using default mode")
        bestMode = "1920x1080x60p"
    end if
    bestModeWidth = ToNumber(bestMode.Tokenize("x")[0])
    bestModeHeight = ToNumber(bestMode.Tokenize("x")[1])

    LogX("Autorun", "Best HDMI mode width: ", bestModeWidth)
    LogX("Autorun", "Best HDMI mode height: ", bestModeHeight)

    activeMode = CreateObject("roVideoMode").getActiveMode()
    LogX("Autorun", "Active mode: ", activeMode)

    ' https://docs.brightsign.biz/developers/rorectangle#dX4DXs
    ' No matter what the active mode is, if it's larger than 1920 & 1080
    ' the device will do upscaling and we need to use the graphics plane width and height instead
    width = activeMode.graphicsPlaneWidth
    height = activeMode.graphicsPlaneHeight
    rect = CreateObject("roRectangle", 0, 0, width, height)

    'Ex. "SD:/"
    defaultDrive = GetDefaultDrive()
    LogX("Autorun", "Default drive:", defaultDrive)

    loadingConfig = {
        url: "file:/" + defaultDrive + "app/renderer/loading.html",
    }
    loading = CreateObject("roHtmlWidget", rect, loadingConfig)
    loading.Show()

    ' set DWS on device
    nc = CreateObject("roNetworkConfiguration", 0)
    if type(nc) <> "roNetworkConfiguration" then
        nc = CreateObject("roNetworkConfiguration", 1)
    endif
    if type(nc) = "roNetworkConfiguration" then
        dwsAA = CreateObject("roAssociativeArray")
        dwsAA["port"] = "80"
        nc.SetupDWS(dwsAA)
        nc.SetTimeServer("pool.ntp.org")
        ' Min 120 seconds
        nc.SetTimeServerIntervalSeconds(3600)
        nc.Apply()
    end if

    ' Wait for 3 second to make sure network interface is ready
    Sleep(3000)

    msgPort = CreateObject("roMessagePort")

    't = CreateObject("roTouchScreen")
    't.SetResolution(width, height)

    ' Use the full screen for testing, including the unsafe area
    't.AddRectangleRegion(0, 0, width, height, 100)
    't.EnableRegion(100, true)
    't.SetCursorBitmap("sys:/default-cursor.png", 15, 16)
    't.SetCursorPosition(width / 2, height / 2)
    't.EnableCursor(1)

    storage = CreateObject("roStorageInfo", defaultDrive)
    LogX("Autorun", "Storage free space:", ToString(storage.GetFreeInMegabytes()) + "MB")

    defaultDriveInJs = LCase(defaultDrive.Tokenize(":")[0])
    defaultDriveInJsArg = "--defaultDrive=" + defaultDriveInJs

    node_arguments = []
    inspector_server = {}

    If NOT IsProduction() Then
        node_arguments = ["--inspect=0.0.0.0:9229"]
        inspector_server = { port: 2999 }
    End If

    nodejs = CreateObject("roNodeJs", defaultDrive + "app/electron/main.js", { message_port: msgPort, node_arguments: node_arguments, arguments: [defaultDriveInJsArg]})

    ' https://docs.brightsign.biz/developers/roregistrysection
    SCRegistry = CreateObject("roRegistrySection", "ScreenCloud")

    transform = SCRegistry.Read("transform")
    if transform = "" then
        transform = "identity"
        SCRegistry.Write("transform", transform)
    end if

    resurrect = "1"
    ' Disable resurrect in non-production
    If NOT IsProduction() Then resurrect = "0"
    SCRegistry.Write("resurrect", resurrect)

    pendingClearCache = SCRegistry.Read("pending_clear_cache")
    if pendingClearCache = "" then
        pendingClearCache = "0"
        SCRegistry.Write("pending_clear_cache", pendingClearCache)
    end if

    LogX("Autorun", "pending_clear_cache: ", pendingClearCache)

    if pendingClearCache = "1" then
        ClearCache()
        SCRegistry.Write("pending_clear_cache", "0")
    end if

    SCRegistry.Flush()

    url = "file:/" + defaultDrive + "app/renderer/index.html"

    LogX("Autorun", "transform: ", transform)
    LogX("Autorun", "url: ", url)

    'https://docs.brightsign.biz/display/DOC/roHtmlWidget
    config = {
        nodejs_enabled: false,
        inspector_server: inspector_server,
        force_unshared_storage: true,
        brightsign_js_objects_enabled: true,
        javascript_enabled: true,
        hwz_default: "on",
        ' To clear cache, we just remove the html_storage folder
        storage_path: defaultDrive + "html_storage",
        ' Use 80% of the free space for the storage quota in bytes
        storage_quota: ToFloat(storage.GetFreeInMegabytes()) * 1024 * 1024 * 0.8,
        security_params: {
            websecurity: false
            insecure_https_enabled: true
            insecure_origin_enabled: true
        },
        port: msgPort,
        transform: transform,
        url: url
    }
    player = CreateObject("roHtmlWidget", rect, config)
    player.SetUserAgentSuffix("ScreenCloud BrightSign/" + GetCurrentScreenCloudBrightSignPlayerVersion())

    maintenanceConfig = {
        nodejs_enabled: true,
        brightsign_js_objects_enabled: true,
        javascript_enabled: true,
        url: "file:/" + defaultDrive + "app/renderer/maintenance.html",
        ' To clear cache, we just remove the html_storage folder
        storage_path: defaultDrive + "html_storage",
        ' Use 80% of the free space for the storage quota in bytes
        storage_quota: ToFloat(storage.GetFreeInMegabytes()) * 1024 * 1024 * 0.8,
        security_params: {
            websecurity: false,
            insecure_https_enabled: true,
            insecure_origin_enabled: true
        },
    }
    maintenance = CreateObject("roHtmlWidget", rect, maintenanceConfig)
    maintenance.EnableJavascript(true)
    maintenance.Hide()

    ' To see print log, use SSH or Telnet
    loop:
        ev = Wait(0, msgPort)
        if type(ev) = "roNodeJsEvent" then
            eventData = ev.GetData()
            if type(eventData) = "roAssociativeArray" and type(eventData.reason) = "roString" then
                if eventData.reason = "process_exit" then
                    LogX("Node.js", "Instance exited with code " + ToString(eventData.exit_code), FormatJson(eventData))
                    ' Resurrect the application if the exit code not 0 and resurrect is enabled
                    if eventData.exit_code <> 1 and resurrect = "1" then
                        LogX("Node.js", "Non 1 exit code, restarting application")
                        RestartApplication()
                    end if
                else if eventData.reason = "message" then
                    ' LogX("Node.js", FormatJson(eventData.message))
                    if eventData.message.action = "started" then
                        LogX("Node.js", "Proxy started", eventData.message.data)
                    else if eventData.message.action = "proxy_config" then
                        player.PostJSMessage({ action: eventData.message.action, data: eventData.message.data, id: eventData.message.id })
                        LogX("Node.js", "Telling player proxy config", eventData.message.data)
                    else if eventData.message.action = "get_setting" then
                        LogX("Node.js", "Telling player get_setting", eventData.message.data)
                        player.PostJSMessage({ action: eventData.message.action, data: eventData.message.data, id: eventData.message.id })
                    else if eventData.message.action = "get_credentials" then
                        LogX("Node.js", "Telling player get_credentials")
                        player.PostJSMessage({ action: eventData.message.action, data: eventData.message.data, id: eventData.message.id })
                    else if eventData.message.action = "set_origin" then
                        LogX("Node.js", "Telling player set_origin finished")
                        player.PostJSMessage({ action: eventData.message.action, id: eventData.message.id })
                    else if eventData.message.action = "get_device_info" then
                        LogX("Node.js", "Telling player get_device_info")
                        player.PostJSMessage({ action: eventData.message.action, data: eventData.message.data, id: eventData.message.id })
                    end if

                else
                    LogX("Node.js", "Unhandled event", eventData.reason)
                end if
                else
                    LogX("Node.js", "Unknown eventData", type(eventData))
                end if
        end if

        if type(ev) = "roHtmlWidgetEvent" then
            payload = ev.GetData()
            if type(payload) = "roAssociativeArray" and type(payload.reason) = "roString" then
                if payload.reason = "message" then
                    if payload.message.action = "set_origin" then
                        LogX("Node.js", "got set_origin", payload.message.data)
                        nodejs.PostJSMessage({ action: payload.message.action, data: payload.message.data, id: payload.message.id })
                    else if payload.message.action = "player_started" then
                        LogX("Player", "got proxy config, showing player")
                        loading.Hide()
                        player.Show()
                    else if payload.message.action = "proxy_config" then
                        LogX("Node.js", "getting proxy_config")
                        nodejs.PostJSMessage({ action: payload.message.action, id: payload.message.id })
                    else if payload.message.action = "get_setting" then
                        LogX("Node.js", "getting setting", payload.message.data)
                        nodejs.PostJSMessage({ action: payload.message.action, data: payload.message.data, id: payload.message.id })
                    else if payload.message.action = "get_credentials" then
                        LogX("Node.js", "getting credentials")
                        nodejs.PostJSMessage({ action: payload.message.action, data: payload.message.data, id: payload.message.id })
                    else if payload.message.action = "save_credentials" then
                        LogX("Node.js", "saving credentials")
                        nodejs.PostJSMessage({ action: payload.message.action, data: payload.message.data, id: payload.message.id })
                    else if payload.message.action = "set_setting" then
                        LogX("Node.js", "setting setting", " " + payload.message.data)
                        nodejs.PostJSMessage({ action: payload.message.action, data: payload.message.data, id: payload.message.id })
                    else if payload.message.action = "check_update" then
                        LogX("Player", "received check_update")
                        CheckReleaseAndUpdate(player,maintenance)
                        player.PostJSMessage({ action: payload.message.action, data: "{}", id: payload.message.id })
                    else if payload.message.action = "test" then
                        LogX("Player", "received test", payload.message.test)
                        ' test send the message back when received the hello message
                        player.PostJSMessage({ test: "hello from brightscript" })
                    else if payload.message.action = "get_current_network_config" then
                        player.PostJSMessage({
                            action: payload.message.action,
                            data: FormatJson(CreateObject("roNetworkConfiguration", 0).GetCurrentConfig())
                            id: payload.message.id
                        })
                    else if payload.message.action = "get_device_info" then
                        LogX("Node.js", "will get_device_info")
                        nodejs.PostJSMessage({ action: payload.message.action, id: payload.message.id })
                    else if payload.message.action = "reboot" then
                        LogX("Player", "BrightScript reboot")
                        ' Worst case if Node.js crashed, we still able to reboot the device with brightscript
                        RebootSystem()
                    else if payload.message.action = "clear_cache" then
                        LogX("Node.js", "will clear_cache")
                        player.Hide()
                        loading.Show()
                        player.FlushCachedResources()
                        SCRegistry.Write("pending_clear_cache", "1")
                        SCRegistry.Flush()
                        RebootSystem()
                    else if payload.message.action = "set_orientation" then
                        LogX("Player", "received set_orientation", payload.message.data)
                        if transform <> payload.message.data then
                            ' We JSON.stringify the data in src/renderer/brightscript.ts
                            SCRegistry.Write("transform", ParseJson(payload.message.data))
                            SCRegistry.Flush()
                            ' Faster than RestartApplication()
                            RestartScript()
                        end if
                    else if payload.message.action = "screenshot" then
                        LogX("Player", "taking screenshot")
                        CreateDirectory("screenshots")
                        screenshot_file = "screenshots/screenshot.jpg"

                        screenshot_result = CreateObject("roVideoMode").Screenshot({
                            filename: "SD:/" + screenshot_file
                            quality: 70,
                            async: 0, ' Wait until finish
                            width: rect.GetWidth(),
                            height: rect.GetHeight()
                        })

                        nodejs.PostJSMessage({ action: payload.message.action, data: payload.message.data, id: payload.message.id })
                    end if
                end if
            else
                LogX("Player", "Unknown payload", type(payload))
            end if
        end if
    goto loop
End Sub
